--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 12.12.2019
Revision: FS25-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Concept/Idea:
Anil

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
]]

FillLevelLimiter = {}

FillLevelLimiter.MOD_NAME = g_currentModName
FillLevelLimiter.SPEC_NAME = string.format("%s.fillLevelLimiter", g_currentModName)
FillLevelLimiter.SPEC_TABLE_NAME = string.format("spec_%s", FillLevelLimiter.SPEC_NAME)

function FillLevelLimiter.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(FillUnit, specializations)
end

function FillLevelLimiter.initSpecialization()
    local schemaSavegame = Vehicle.xmlSchemaSavegame
    local baseKey = string.format("vehicles.vehicle(?).%s", FillLevelLimiter.SPEC_NAME)

    schemaSavegame:register(XMLValueType.BOOL, baseKey .. "#showHelp", "Help information state")
    schemaSavegame:register(XMLValueType.INT, baseKey .. ".unit(?)#index", "Fill Unit index")
    schemaSavegame:register(XMLValueType.FLOAT, baseKey .. ".unit(?)#limit", "Set limit")
end

function FillLevelLimiter.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "addFillUnitFillLevel", FillLevelLimiter.addFillUnitFillLevel)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getFillUnitFreeCapacity", FillLevelLimiter.getFillUnitFreeCapacity)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "showInfo", FillLevelLimiter.showInfo)
end

function FillLevelLimiter.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", FillLevelLimiter)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", FillLevelLimiter)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", FillLevelLimiter)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream", FillLevelLimiter)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", FillLevelLimiter)
    SpecializationUtil.registerEventListener(vehicleType, "onReadUpdateStream", FillLevelLimiter)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteUpdateStream", FillLevelLimiter)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", FillLevelLimiter)
end

function FillLevelLimiter.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "getFillUnitLimit", FillLevelLimiter.getFillUnitLimit)
    SpecializationUtil.registerFunction(vehicleType, "setFillUnitLimit", FillLevelLimiter.setFillUnitLimit)
    SpecializationUtil.registerFunction(vehicleType, "updateFillUnitLimits", FillLevelLimiter.updateFillUnitLimits)
end

function FillLevelLimiter:onLoad(savegame)
    self.spec_fillLevelLimiter = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if self.spec_fillLevelLimiter == nil then
        Logging:error("[%s] Specialization with name 'fillLevelLimiter' was not found in modDesc!", FillLevelLimiter.MOD_NAME)
    end

    local spec = self.spec_fillLevelLimiter

    local fillUnitIndexToLimit = {}
    local numFillUnitLimits = 0

    local fillUnitTitle = {}
    local consumerFillUnitIndexs = {}

    local motorizedSpec = self.spec_motorized

    if motorizedSpec ~= nil and motorizedSpec.consumersByFillTypeName ~= nil then
        for _, consumer in pairs(motorizedSpec.consumersByFillTypeName) do
            consumerFillUnitIndexs[consumer.fillUnitIndex] = consumer
        end
    end

    local levelerSpec = self.spec_leveler

    for fillUnitIndex, _ in ipairs(self.spec_fillUnit.fillUnits) do
        if consumerFillUnitIndexs[fillUnitIndex] == nil and fillUnitIndexToLimit[fillUnitIndex] == nil then
            if levelerSpec == nil or levelerSpec.fillUnitIndex ~= fillUnitIndex then
                local defaultLimit = self:getFillUnitCapacity(fillUnitIndex)

                if defaultLimit ~= nil then
                    fillUnitIndexToLimit[fillUnitIndex] = defaultLimit
                    numFillUnitLimits = numFillUnitLimits + 1
                end
            end
        end
    end

    local i = 1

    for fillUnitIndex, _ in pairs(fillUnitIndexToLimit) do
        fillUnitTitle[fillUnitIndex] = FillLevelLimiter.getFillUnitTitle(self, fillUnitIndex, i, numFillUnitLimits)

        i = i + 1
    end

    spec.showHelpInfo = true

    spec.fillUnitIndexToLimit = fillUnitIndexToLimit
    spec.numFillUnitLimits = numFillUnitLimits

    spec.fillUnitTitle = fillUnitTitle
    spec.consumerFillUnitIndexs = consumerFillUnitIndexs

    spec.fillLimitReachedWarning = 0
    spec.limitText = g_i18n:getText("info_fll_limit", FillLevelLimiter.MOD_NAME)

    spec.dirtyFlag = self:getNextDirtyFlag()
end

function FillLevelLimiter:onPostLoad(savegame)
    local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if spec.numFillUnitLimits > 0 then
        if self.isServer and savegame ~= nil then
            local xmlFile = savegame.xmlFile
            local savegameKey = savegame.key .. "." .. FillLevelLimiter.SPEC_NAME

            if xmlFile:hasProperty(savegameKey) then
                local showHelp = xmlFile:getValue(savegameKey .. "#showHelp", true)
                local i = 0

                while true do
                    local key = string.format("%s.unit(%d)", savegameKey, i)

                    if not xmlFile:hasProperty(key) then
                        break
                    end

                    local fillUnitIndex = xmlFile:getValue(key .. "#index", 0)

                    if spec.fillUnitIndexToLimit[fillUnitIndex] ~= nil then
                        local fillLimit = xmlFile:getValue(key .. "#limit")

                        if fillLimit ~= nil then
                            self:setFillUnitLimit(fillUnitIndex, fillLimit)
                        end
                    end

                    i = i + 1
                end
            end
        end
    else
        SpecializationUtil.removeEventListener(self, "onUpdate", FillLevelLimiter)
        SpecializationUtil.removeEventListener(self, "onReadStream", FillLevelLimiter)
        SpecializationUtil.removeEventListener(self, "onWriteStream", FillLevelLimiter)
        SpecializationUtil.removeEventListener(self, "onReadUpdateStream", FillLevelLimiter)
        SpecializationUtil.removeEventListener(self, "onWriteUpdateStream", FillLevelLimiter)
        SpecializationUtil.removeEventListener(self, "onRegisterActionEvents", FillLevelLimiter)
    end
end

function FillLevelLimiter:saveToXMLFile(xmlFile, key, usedModNames)
    local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if spec.numFillUnitLimits > 0 then
        if not spec.showHelpInfo then
            xmlFile:setValue(key .. "#showHelp", spec.showHelpInfo)
        end

        local i = 0

        for fillUnitIndex, fillLimit in pairs(spec.fillUnitIndexToLimit) do
            if fillLimit < (self:getFillUnitCapacity(fillUnitIndex) or 0) then
                local fillUnitKey = string.format("%s.unit(%d)", key, i)

                xmlFile:setValue(fillUnitKey .. "#index", fillUnitIndex)
                xmlFile:setValue(fillUnitKey .. "#limit", fillLimit)

                i = i + 1
            end
        end
    end
end

function FillLevelLimiter:onReadStream(streamId, connection)
    if connection:getIsServer() then
        local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

        local numLimits = streamReadInt8(streamId)

        if numLimits > 0 then
            spec.showHelpInfo = streamReadBool(streamId)

            for i = 1, numLimits do
                local fillUnitIndex = streamReadUIntN(streamId, 8)
                local limit = streamReadFloat32(streamId)

                self:setFillUnitLimit(fillUnitIndex, limit)
            end
        end
    end
end

function FillLevelLimiter:onWriteStream(streamId, connection)
    if not connection:getIsServer() then
        local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

        local numLimits = spec.numFillUnitLimits
        streamWriteInt8(streamId, numLimits)

        if numLimits > 0 then
            streamWriteBool(streamId, spec.showHelpInfo)

            for fillUnitIndex, limit in pairs (spec.fillUnitIndexToLimit) do
                streamWriteUIntN(streamId, fillUnitIndex, 8)
                streamWriteFloat32(streamId, limit)
            end
        end
    end
end

function FillLevelLimiter:onReadUpdateStream(streamId, timestamp, connection)
    if connection:getIsServer() then
        if streamReadBool(streamId) then
            local fillLimitReachedWarning = streamReadUIntN(streamId, 8)
            FillLevelLimiter.displayLimitReachedWarning(self, fillLimitReachedWarning)
        end
    end
end

function FillLevelLimiter:onWriteUpdateStream(streamId, connection, dirtyMask)
    if not connection:getIsServer() then
        local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

        if streamWriteBool(streamId, bitAND(dirtyMask, spec.dirtyFlag) ~= 0) then
            streamWriteUIntN(streamId, spec.fillLimitReachedWarning or 0, 8)
            spec.fillLimitReachedWarning = 0
        end
    end
end

function FillLevelLimiter:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if isActiveForInput then
        if spec.showHelpInfo and spec.numFillUnitLimits > 0 then
            local fillUnits = self.spec_fillUnit.fillUnits

            for fillUnitIndex, limit in pairs (spec.fillUnitIndexToLimit) do
                local capacity = self:getFillUnitCapacity(fillUnitIndex)

                if capacity ~= nil and limit < capacity then
                    local title = spec.fillUnitTitle[fillUnitIndex]

                    if title ~= nil then
                        local fillUnit = fillUnits[fillUnitIndex]
                        local unitText = (fillUnit ~= nil and fillUnit.unitText) or g_i18n:getVolumeUnit()

                        g_currentMission:addExtraPrintText(string.format("%s (%s):  %d %s", spec.limitText, title, math.floor(limit + 0.5), unitText))
                    end
                end
            end
        end
    end
end

function FillLevelLimiter:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if self.isClient then
        local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

        self:clearActionEventsTable(spec.actionEvents)

        if isActiveForInput and spec.numFillUnitLimits > 0 then
            local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.SET_VEHICLE_FILL_LIMIT, self, FillLevelLimiter.actionEventOpenUI, false, true, false, true, nil)

            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)
        end
    end
end

function FillLevelLimiter:setFillUnitLimit(fillUnitIndex, limit)
    local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if spec ~= nil and fillUnitIndex ~= nil and spec.fillUnitIndexToLimit[fillUnitIndex] ~= nil then
        local maxLimit = self:getFillUnitCapacity(fillUnitIndex) or 0

        if limit == nil then
            limit = maxLimit
        end

        -- spec.fillUnitIndexToLimit[fillUnitIndex] = math.clamp(limit, 1, maxLimit)
        spec.fillUnitIndexToLimit[fillUnitIndex] = math.min(math.max(limit, 1), maxLimit)
    end
end

function FillLevelLimiter:updateFillUnitLimits(limitsTable, showHelpInfo, noEventSend)
    local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if spec ~= nil then
        showHelpInfo = Utils.getNoNil(showHelpInfo, spec.showHelpInfo)

        FillLevelLimiterEvent.sendEvent(self, limitsTable, showHelpInfo, noEventSend)

        if limitsTable ~= nil then
            for fillUnitIndex, limit in pairs (limitsTable) do
                if limit ~= spec.fillUnitIndexToLimit[fillUnitIndex] then
                    self:setFillUnitLimit(fillUnitIndex, limit)
                end
            end
        end

        spec.showHelpInfo = showHelpInfo
    end
end

function FillLevelLimiter:getFillUnitLimit(fillUnitIndex)
    local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if spec ~= nil and spec.numFillUnitLimits > 0 and fillUnitIndex ~= nil then
        return spec.fillUnitIndexToLimit[fillUnitIndex]
    end

    return nil
end

function FillLevelLimiter:addFillUnitFillLevel(superFunc, farmId, fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData)
    local fillUnitLimit = self:getFillUnitLimit(fillUnitIndex)
    local fillUnit = self.spec_fillUnit.fillUnits[fillUnitIndex]

    if fillUnitLimit ~= nil and fillUnit ~= nil and fillUnitLimit < fillUnit.capacity then
        if fillUnit.fillLevel + fillLevelDelta >= fillUnitLimit then
            fillLevelDelta = fillUnitLimit - fillUnit.fillLevel

            if self.isServer then
                local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

                if spec ~= nil then
                    FillLevelLimiter.displayLimitReachedWarning(self, fillUnitIndex)

                    spec.fillLimitReachedWarning = fillUnitIndex
                    self:raiseDirtyFlags(spec.dirtyFlag)
                end
            end
        end
    end

    return superFunc(self, farmId, fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData)
end

function FillLevelLimiter:getFillUnitFreeCapacity(superFunc, fillUnitIndex, fillTypeIndex, farmId)
    local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if spec == nil or spec.numFillUnitLimits == 0 then
        return superFunc(self, fillUnitIndex, fillTypeIndex, farmId)
    end

    local fillUnit = self.spec_fillUnit.fillUnits[fillUnitIndex]

    if fillUnit == nil then
        return nil
    end

    if not fillUnit.ignoreFillLimit and g_currentMission.missionInfo.trailerFillLimit and self:getMaxComponentMassReached() then
        return 0
    end

    local fillUnitLimit = self:getFillUnitLimit(fillUnitIndex)

    if fillUnitLimit ~= nil and fillUnitLimit < fillUnit.capacity then
        return fillUnitLimit - fillUnit.fillLevel
    end

    return fillUnit.capacity - fillUnit.fillLevel
end

function FillLevelLimiter:showInfo(superFunc, box)
    local spec = self[FillLevelLimiter.SPEC_TABLE_NAME]

    if spec ~= nil and spec.numFillUnitLimits > 0 then
        for fillUnitIndex, limit in pairs (spec.fillUnitIndexToLimit) do
            local capacity = self:getFillUnitCapacity(fillUnitIndex)

            if capacity ~= nil and limit < capacity then
                local title = spec.fillUnitTitle[fillUnitIndex]

                if title ~= nil then
                    local fillUnit = self.spec_fillUnit.fillUnits[fillUnitIndex]
                    local unitText = (fillUnit ~= nil and fillUnit.unitText) or g_i18n:getVolumeUnit()

                    box:addLine(string.format("%s (%s)", spec.limitText, title), string.format("%d %s", math.floor(limit + 0.5), unitText))
                end
            end
        end
    end

    superFunc(self, box)
end

function FillLevelLimiter.actionEventOpenUI(self)
    FillLevelLimiterDialog.show(self.updateFillUnitLimits, self, self:getFullName(), self, self[FillLevelLimiter.SPEC_TABLE_NAME].showHelpInfo)
end

function FillLevelLimiter.displayLimitReachedWarning(self, fillUnitIndex)
    if fillUnitIndex ~= nil and fillUnitIndex > 0 then
        local displayWarning = self:getIsActiveForInput()
        local isExternal = false

        if not displayWarning and self.spec_mixerWagon ~= nil then
            local inputHelp = g_currentMission.hud.inputHelp

            if inputHelp ~= nil and inputHelp.infoExtensions ~= nil then
                for _, infoExtension in pairs (inputHelp.infoExtensions) do
                    if infoExtension.vehicle == self then
                        displayWarning = true
                        isExternal = true

                        break
                    end
                end
            end
        end

        if displayWarning then
            local limit = self:getFillUnitLimit(fillUnitIndex)

            if limit ~= nil then
                local fillUnit = self.spec_fillUnit.fillUnits[fillUnitIndex]
                local unitText = (fillUnit ~= nil and fillUnit.unitText) or g_i18n:getVolumeUnit(false)

                local text = string.format(g_i18n:getText("warning_fll_fillLimitReached", FillLevelLimiter.MOD_NAME), limit, unitText)
                local duration = 2000

                if isExternal then
                    text = string.format("%s %s", self:getFullName(), text)
                    duration = 3500
                end

                g_currentMission:showBlinkingWarning(text, duration, 1)
            end
        end
    end
end

function FillLevelLimiter.getFillUnitTitle(vehicle, fillUnitIndex, index, numFillUnitLimits)
    numFillUnitLimits = numFillUnitLimits or 1

    local sowingMachineSpec = vehicle.spec_sowingMachine

    if sowingMachineSpec ~= nil then
        if sowingMachineSpec.fillUnitIndex == fillUnitIndex then
            return g_fillTypeManager:getFillTypeTitleByIndex(FillType.SEEDS)
        else
            local sprayerSpec = vehicle.spec_sprayer

            if sprayerSpec ~= nil and sprayerSpec.fillUnitIndex == fillUnitIndex then
                local fillTypeIndex = FillType.FERTILIZER

                if sprayerSpec.supportedSprayTypes ~= nil then
                    fillTypeIndex = sprayerSpec.supportedSprayTypes[1] or fillTypeIndex
                end

                return g_fillTypeManager:getFillTypeTitleByIndex(fillTypeIndex)
            end
        end
    end

    if numFillUnitLimits > 1 and (vehicle.xmlFile ~= nil and vehicle.xmlFile.handle ~= nil) then
        local fillUnitConfigurationId = Utils.getNoNil(vehicle.configurations.fillUnit, 1)
        local key = string.format("vehicle.fillUnit.fillUnitConfigurations.fillUnitConfiguration(%d).fillUnits.fillUnit(%d)", fillUnitConfigurationId - 1, fillUnitIndex - 1)

        if vehicle.xmlFile:hasProperty(key .. ".exactFillRootNode") then
            local id = getXMLString(vehicle.xmlFile.handle, key .. ".exactFillRootNode#node")

            if id ~= nil and vehicle.i3dMappings[id] ~= nil then
                id = id:upper()

                if id:find("FRONT") then
                    return g_i18n:getText("info_tipSideFront")
                elseif id:find("MIDDLE") then
                    return g_i18n:getText("info_tipSideMiddle")
                elseif id:find("BACK") or id:find("REAR") then
                    return g_i18n:getText("info_tipSideBack")
                end
            end
        end
    end

    return tostring(index or "Unknown")
end
